<?php
declare(strict_types=1);

/**
 * modules/corporates/delete.php
 *
 * Delete a Corporate Account.
 *
 * Parts:
 *  1) Bootstrap & guards
 *  2) Fetch target corporate (for confirmation)
 *  3) Handle POST (delete & redirect)
 *  4) Render confirmation UI
 */

////////////////////////////////////////////////////////////////
// 1) Bootstrap & guards
////////////////////////////////////////////////////////////////
require_once dirname(__DIR__, 2) . '/config/functions.php';
require_role(['Admin','Ops','sales']);

$user = current_user();
$cid  = (int)($user['company_id'] ?? 0);
if ($cid <= 0) redirect(url_public('login.php'));

$id = (int)($_GET['id'] ?? $_POST['id'] ?? 0);
if ($id <= 0) redirect(url_modules('corporates/list.php'));

$errors = [];
$notice = null;

////////////////////////////////////////////////////////////////
// 2) Fetch target corporate (for confirmation)
////////////////////////////////////////////////////////////////
$sel = db()->prepare("
  SELECT *
    FROM corporates
   WHERE id = :id AND company_id = :cid
   LIMIT 1
");
$sel->execute([':id' => $id, ':cid' => $cid]);
$corp = $sel->fetch();
if (!$corp) {
  // Already gone or wrong company
  redirect(url_modules('corporates/list.php'));
}

////////////////////////////////////////////////////////////////
// 3) Handle POST (delete & redirect)
////////////////////////////////////////////////////////////////
if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete') {
  try {
    csrf_verify((string)($_POST['csrf'] ?? ''));

    $del = db()->prepare("DELETE FROM corporates WHERE id = :id AND company_id = :cid LIMIT 1");
    $del->execute([':id' => $id, ':cid' => $cid]);

    audit_log('corporate.delete', 'corporate', $id, ['name' => $corp['name'] ?? null]);

    redirect(url_modules('corporates/list.php') . '?deleted=1');
  } catch (Throwable $e) {
    $errors[] = (defined('APP_ENV') && APP_ENV === 'dev')
      ? $e->getMessage()
      : 'Unable to delete this corporate account.';
  }
}

////////////////////////////////////////////////////////////////
// 4) Render confirmation UI
////////////////////////////////////////////////////////////////
include dirname(__DIR__, 2) . '/includes/header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 mb-0">Delete Corporate Account</h1>
  <a href="<?= e(url_modules('corporates/list.php')) ?>" class="btn btn-outline-secondary">Back to List</a>
</div>

<?php if ($errors): ?>
  <div class="alert alert-danger"><ul class="mb-0"><?php foreach ($errors as $er) echo '<li>'.e($er).'</li>'; ?></ul></div>
<?php endif; ?>

<div class="card shadow-sm">
  <div class="card-body">
    <h2 class="h6 text-uppercase text-muted">Confirm Delete</h2>
    <p class="mb-3">
      You are about to permanently remove the following corporate account:
    </p>

    <dl class="row">
      <dt class="col-sm-3">Company Name</dt>
      <dd class="col-sm-9"><?= e($corp['name'] ?? '—') ?></dd>

      <dt class="col-sm-3">Contact</dt>
      <dd class="col-sm-9"><?= e($corp['contact_person'] ?? '—') ?></dd>

      <dt class="col-sm-3">Email</dt>
      <dd class="col-sm-9"><?= e($corp['email'] ?? '—') ?></dd>

      <dt class="col-sm-3">Phone</dt>
      <dd class="col-sm-9"><?= e($corp['phone'] ?? '—') ?></dd>
    </dl>

    <div class="alert alert-warning">
      <strong>Warning:</strong> This action cannot be undone.
      If this corporate has related bookings/invoices, consider deactivating instead of deleting.
    </div>

    <form method="post" class="d-flex gap-2 mt-3">
      <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
      <input type="hidden" name="id" value="<?= (int)$id ?>">
      <input type="hidden" name="action" value="delete">
      <button class="btn btn-danger" onclick="return confirm('Permanently delete this corporate account?')">
        Yes, Delete
      </button>
      <a class="btn btn-outline-secondary" href="<?= e(url_modules('corporates/list.php')) ?>">Cancel</a>
    </form>
  </div>
</div>

<?php include dirname(__DIR__, 2) . '/includes/footer.php'; ?>
