<?php
declare(strict_types=1);
/**
 * modules/bookings/list.php
 *
 * Feature highlights
 * - Filters: Client (text), Booking Ref (last digits ok), Driver, Partner, Corporate, Date range, View (Active / Cleared / All)
 * - Compact, scrollable table with fixed column order
 * - Assignment highlight (green) when at least one driver/vehicle row exists
 * - Copy-to-clipboard for summary & driver phones
 * - Profit = (client + client extras) - (driver + driver extras) - commission
 * - Works with driver “own vehicle” (free-text reg/type) or fleet vehicle
 */
require_once dirname(__DIR__, 2) . '/config/functions.php';
require_role(['Admin','Ops','MD']);

$user = current_user();
$cid  = (int)($user['company_id'] ?? 0);
if ($cid <= 0) redirect(url_public('login.php'));

/* ---------------- Lookups ---------------- */
$driversStmt = db()->prepare("SELECT id, name FROM drivers WHERE company_id=:cid AND is_active=1 ORDER BY name");
$driversStmt->execute([':cid'=>$cid]);
$drivers = $driversStmt->fetchAll();

$partnersStmt = db()->prepare("SELECT id, name FROM partners WHERE company_id=:cid ORDER BY name");
$partnersStmt->execute([':cid'=>$cid]);
$partners = $partnersStmt->fetchAll();

$corporatesStmt = db()->prepare("
  SELECT id, name
  FROM corporates
  WHERE company_id=:cid AND status='active'
  ORDER BY name
");
$corporatesStmt->execute([':cid'=>$cid]);
$corporates = $corporatesStmt->fetchAll();

/* ---------------- Filters ---------------- */
$view          = ($_GET['view'] ?? 'active'); // active, cleared, all
$q_client      = trim((string)($_GET['client'] ?? ''));
$q_ref         = trim((string)($_GET['ref'] ?? ''));     // booking ref (suffix digits OK)
$q_driver_id   = (int)($_GET['driver_id'] ?? 0);
$q_partner     = (int)($_GET['partner_id'] ?? 0);
$q_corporate   = (int)($_GET['corporate_id'] ?? 0);
$q_from        = trim((string)($_GET['from'] ?? ''));
$q_to          = trim((string)($_GET['to'] ?? ''));

$where   = ["b.company_id = :cid"];
$params  = [':cid' => $cid];

if ($view === 'active') {
  $where[] = "b.status IN ('New','Confirmed','Assigned','InProgress')";
  $order   = "b.pickup_date ASC, b.pickup_time ASC, b.id ASC";
} elseif ($view === 'cleared') {
  $where[] = "b.status = 'Completed'";
  $order   = "b.pickup_date DESC, b.pickup_time DESC, b.id DESC";
} else {
  $order   = "b.pickup_date DESC, b.pickup_time DESC, b.id DESC";
}

if ($q_client !== '') {
  $where[] = "b.client_name LIKE :client";
  $params[':client'] = "%{$q_client}%";
}

/* Booking ref filter: allow last digits as suffix match */
if ($q_ref !== '') {
  if (preg_match('/^\d+$/', $q_ref)) {
    $where[] = "b.booking_ref LIKE :ref_like";
    $params[':ref_like'] = "%{$q_ref}";
  } else {
    $where[] = "b.booking_ref LIKE :ref_like";
    $params[':ref_like'] = "%{$q_ref}%";
  }
}

if ($q_partner > 0) {
  $where[] = "b.partner_id = :pid";
  $params[':pid'] = $q_partner;
}

if ($q_corporate > 0) {
  $where[] = "b.corporate_id = :corp";
  $params[':corp'] = $q_corporate;
}

if ($q_driver_id > 0) {
  // Filter bookings that have at least one row with driver_id = :did
  $where[] = "EXISTS (
    SELECT 1 FROM booking_vehicles bv
    WHERE bv.booking_id = b.id AND bv.driver_id = :did
  )";
  $params[':did'] = $q_driver_id;
}

if ($q_from !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $q_from)) {
  $where[] = "b.pickup_date >= :from";
  $params[':from'] = $q_from;
}
if ($q_to !== '' && preg_match('/^\d{4}-\d{2}-\d{2}$/', $q_to)) {
  $where[] = "b.pickup_date <= :to";
  $params[':to'] = $q_to;
}

$whereSql = implode(' AND ', $where);

/* ---------------- Data query ---------------- */
$sql = "
  SELECT
    b.id, b.booking_ref, b.booking_type, b.status,
    b.client_name, b.client_phone,
    b.pickup_date, b.pickup_time, b.pickup_address, b.dropoff_address,
    b.total_client_price, b.total_driver_price,
    b.client_parking_fee, b.client_waiting_fee, b.driver_parking_fee, b.driver_waiting_fee,
    b.partner_commission_amount,
    b.pax_count, b.luggage_count,
    p.name AS partner_name,
    c.name AS corporate_name
  FROM bookings b
  LEFT JOIN partners  p ON p.id = b.partner_id
  LEFT JOIN corporates c ON c.id = b.corporate_id
  WHERE {$whereSql}
  ORDER BY {$order}
  LIMIT 200
";
$stmt = db()->prepare($sql);
$stmt->execute($params);
$rows = $stmt->fetchAll();

/* ---------------- Assignment map for visible rows ---------------- */
function fetch_assignments_map(array $bookingRows): array {
  if (!$bookingRows) return [[], []];
  $ids = array_map(fn($r) => (int)$r['id'], $bookingRows);
  $ids = array_values(array_unique(array_filter($ids)));
  if (!$ids) return [[], []];
  $in = implode(',', array_map('intval', $ids));
  $sql = "
    SELECT
      bv.booking_id,
      d.name  AS driver_name,
      d.phone AS driver_phone,
      COALESCE(bv.vehicle_reg_no, v.reg_no) AS reg_no,
      COALESCE(bv.vehicle_type,   v.type)   AS vehicle_type,
      bv.sequence_order
    FROM booking_vehicles bv
    LEFT JOIN drivers  d ON d.id=bv.driver_id
    LEFT JOIN vehicles v ON v.id=bv.vehicle_id
    WHERE bv.booking_id IN ($in)
    ORDER BY bv.sequence_order ASC, bv.id ASC
  ";
  $q = db()->query($sql);
  $map = []; $phones = [];
  while ($a = $q->fetch()) {
    $bid = (int)$a['booking_id'];
    $map[$bid][] = $a;
    if (!empty($a['driver_phone'])) $phones[$bid][] = $a['driver_phone'];
  }
  return [$map, $phones];
}
[$assignMap, $phoneMap] = fetch_assignments_map($rows);

/* ---------------- Helpers ---------------- */
function status_badge_row(string $s): string {
  $map = [
    'New'        => 'secondary',
    'Confirmed'  => 'primary',
    'Assigned'   => 'info',
    'InProgress' => 'warning',
    'Completed'  => 'secondary',
    'Invoiced'   => 'info',
    'Paid'       => 'success',
  ];
  $cls = $map[$s] ?? 'secondary';
  return '<span class="badge text-bg-'.e($cls).'">'.e($s).'</span>';
}

function build_assignment_lines(array $as): string {
  if (!$as) return '<span class="text-muted">—</span>';
  $h = '<ul class="list-unstyled mb-1 small">';
  foreach ($as as $x) {
    $line = '<strong>'.e($x['driver_name'] ?: '—').'</strong>';
    if (!empty($x['reg_no'])) {
      $line .= '<div class="text-muted">'.e($x['reg_no']);
      if (!empty($x['vehicle_type'])) $line .= ' ('.e($x['vehicle_type']).')';
      $line .= '</div>';
    }
    if (!empty($x['driver_phone'])) {
      $line .= '<div class="d-flex gap-1">'.
                 '<a class="btn btn-sm btn-outline-secondary btn-icon" href="tel:'.e($x['driver_phone']).'">Call</a>'.
                 '<button type="button" class="btn btn-sm btn-outline-secondary btn-icon js-copy" data-copy="'.e($x['driver_phone']).'">📋</button>'.
               '</div>';
    }
    $h .= '<li class="mb-1">'.$line.'</li>';
  }
  $h .= '</ul>';
  if (!empty($as)) {
    $all = array_values(array_filter(array_column($as,'driver_phone')));
    if ($all) $h .= '<button type="button" class="btn btn-sm btn-outline-secondary js-copy" data-copy="'.e(implode(', ', $all)).'">Copy All Phones</button>';
  }
  return $h;
}

/* ---------------- View ---------------- */
include dirname(__DIR__, 2) . '/includes/header.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 mb-0">Bookings</h1>
  <div class="d-flex gap-2">
    <a href="<?= e(url_modules('bookings/add.php')) ?>" class="btn btn-dark">Add Booking</a>
  </div>
</div>

<div class="card shadow-sm mb-3">
  <div class="card-body">
    <form class="row g-2 align-items-end" method="get" autocomplete="off">
      <div class="col-6 col-md-2">
        <label class="form-label">View</label>
        <select class="form-select" name="view">
          <option value="active"  <?= $view==='active'?'selected':'' ?>>Active</option>
          <option value="cleared" <?= $view==='cleared'?'selected':'' ?>>Cleared</option>
          <option value="all"     <?= $view==='all'?'selected':'' ?>>All</option>
        </select>
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">Client</label>
        <input class="form-control" name="client" value="<?= e($q_client) ?>" placeholder="Type a name">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">Ref (end digits ok)</label>
        <input class="form-control" name="ref" value="<?= e($q_ref) ?>" placeholder="e.g. 512">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">Driver</label>
        <select class="form-select" name="driver_id">
          <option value="">—</option>
          <?php foreach ($drivers as $d): ?>
            <option value="<?= (int)$d['id'] ?>" <?= $q_driver_id===(int)$d['id']?'selected':'' ?>><?= e($d['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">Partner</label>
        <select class="form-select" name="partner_id">
          <option value="">—</option>
          <?php foreach ($partners as $p): ?>
            <option value="<?= (int)$p['id'] ?>" <?= $q_partner===(int)$p['id']?'selected':'' ?>><?= e($p['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">Corporate</label>
        <select class="form-select" name="corporate_id">
          <option value="">—</option>
          <?php foreach ($corporates as $c): ?>
            <option value="<?= (int)$c['id'] ?>" <?= $q_corporate===(int)$c['id']?'selected':'' ?>><?= e($c['name']) ?></option>
          <?php endforeach; ?>
        </select>
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">From</label>
        <input type="date" class="form-control" name="from" value="<?= e($q_from) ?>">
      </div>
      <div class="col-6 col-md-2">
        <label class="form-label">To</label>
        <input type="date" class="form-control" name="to" value="<?= e($q_to) ?>">
      </div>
      <div class="col-12 col-md-4">
        <button class="btn btn-primary">Apply Filters</button>
        <a class="btn btn-outline-secondary" href="<?= e(url_modules('bookings/list.php')) ?>">Reset</a>
      </div>
    </form>
  </div>
</div>

<div class="card shadow-sm">
  <div class="card-body">
    <div class="table-responsive" style="overflow-x:auto;">
      <table class="table align-middle mb-0 bookings-table">
        <thead class="table-light">
          <tr>
            <th style="min-width:160px;">Ref</th>
            <th>Client</th>
            <th style="min-width:220px;">Pickup</th>
            <th style="min-width:220px;">Dropoff</th>
            <th>Status</th>
            <th>Partner</th>
            <th>Corporate</th>
            <th style="min-width:260px;">Assignment</th>
            <th class="text-end">£ Client</th>
            <th class="text-end">£ Driver</th>
            <th class="text-end">£ C.Extras</th>
            <th class="text-end">£ D.Extras</th>
            <th class="text-end">£ Profit</th>
            <th>Type</th>
          </tr>
        </thead>
        <tbody>
        <?php if ($rows): foreach ($rows as $r):
          $bid = (int)$r['id'];
          $as  = $assignMap[$bid] ?? [];
          $phones = $phoneMap[$bid] ?? [];
          $allPhones = $phones ? implode(', ', $phones) : '';

          $client  = (float)$r['total_client_price'];
          $driver  = (float)$r['total_driver_price'];
          $cExtras = (float)$r['client_parking_fee'] + (float)$r['client_waiting_fee'];
          $dExtras = (float)$r['driver_parking_fee'] + (float)$r['driver_waiting_fee'];
          $commission = (float)($r['partner_commission_amount'] ?? 0);
          $profit  = ($client + $cExtras) - ($driver + $dExtras) - $commission;
          $pBadge  = $profit > 0 ? 'success' : ($profit < 0 ? 'danger' : 'secondary');

          $pickupDT = trim(($r['pickup_date'] ?: '') . ' ' . ($r['pickup_time'] ?: ''));
          $summary = "Booking: ".$r['booking_ref']
            ."\nDate/Time: ".$pickupDT
            ."\nClient: ".($r['client_name'] ?: '—')." (".($r['client_phone'] ?: '—').")"
            ."\nPartner: ".($r['partner_name'] ?: '—')
            ."\nCorporate: ".($r['corporate_name'] ?: '—')
            ."\nRoute: ".($r['pickup_address'] ?: '—')." -> ".($r['dropoff_address'] ?: '—')
            ."\nVehicles: ".implode(' | ', array_map(function($x){
                $s = $x['driver_name'] ?: '—';
                if (!empty($x['reg_no'])) $s .= ' — '.$x['reg_no'].(!empty($x['vehicle_type']) ? ' ('.$x['vehicle_type'].')' : '');
                return $s;
              }, $as))
            ."\nPax/Luggage: ".(int)$r['pax_count']." / ".(int)$r['luggage_count'];
          ?>
          <tr class="<?= ($as && $r['status']!=='Completed') ? 'table-success' : '' ?>">
            <td class="align-top">
              <div><strong><?= e($r['booking_ref']) ?></strong></div>
              <div class="d-flex flex-wrap gap-1 mt-1">
                <button type="button" class="btn btn-sm btn-outline-secondary js-copy" data-copy="<?= e($summary) ?>">Copy</button>
                <a class="btn btn-sm btn-outline-secondary" href="<?= e(url_modules('bookings/pdf.php')) . '?booking_id=' . $bid ?>">Journey</a>
                <a class="btn btn-sm btn-outline-primary" href="<?= e(url_modules('bookings/edit.php')) . '?id=' . $bid ?>">Edit</a>
                <?php if ($r['status'] !== 'Completed'): ?>
                <form method="post" action="<?= e(url_modules('bookings/complete.php')) ?>" class="d-inline">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= $bid ?>">
                  <button class="btn btn-sm btn-outline-success">Complete</button>
                </form>
                <?php endif; ?>
                <form method="post" action="<?= e(url_modules('bookings/delete.php')) ?>" class="d-inline">
                  <input type="hidden" name="csrf" value="<?= e(csrf_token()) ?>">
                  <input type="hidden" name="id" value="<?= $bid ?>">
                  <button class="btn btn-sm btn-outline-danger" onclick="return confirm('Delete this booking?')">Delete</button>
                </form>
              </div>
            </td>
            <td class="text-wrap">
              <?= e($r['client_name'] ?: '—') ?>
              <?php if ($r['client_phone']): ?><br><small class="text-muted"><?= e($r['client_phone']) ?></small><?php endif; ?>
            </td>
            <td class="text-wrap"><?= e($pickupDT . ($r['pickup_address'] ? ' | ' . $r['pickup_address'] : '')) ?></td>
            <td class="text-wrap"><?= e($r['dropoff_address'] ?: '—') ?></td>
            <td><?= status_badge_row($r['status']) ?></td>
            <td><?= e($r['partner_name'] ?: '—') ?></td>
            <td><?= e($r['corporate_name'] ?: '—') ?></td>
            <td class="text-wrap"><?= build_assignment_lines($as) ?></td>
            <td class="text-end"><?= number_format($client, 2) ?></td>
            <td class="text-end"><?= number_format($driver, 2) ?></td>
            <td class="text-end"><?= number_format($cExtras, 2) ?></td>
            <td class="text-end"><?= number_format($dExtras, 2) ?></td>
            <td class="text-end"><span class="badge text-bg-<?= e($pBadge) ?>"><?= number_format($profit, 2) ?></span></td>
            <td><?= e($r['booking_type']) ?></td>
          </tr>
        <?php endforeach; else: ?>
          <tr><td colspan="14" class="text-center text-muted py-4">No bookings found for the selected filters.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
    </div>
  </div>
</div>

<?php include dirname(__DIR__, 2) . '/includes/footer.php'; ?>

<script>
// Copy-to-clipboard for any .js-copy button
document.addEventListener('click', function(e){
  const btn = e.target.closest('.js-copy');
  if (!btn) return;
  const text = btn.getAttribute('data-copy') || '';
  navigator.clipboard.writeText(text).then(()=>{
    toast('Copied');
  });
});
function toast(text){
  const t = document.createElement('div');
  t.textContent = text;
  t.style.position='fixed';
  t.style.right='1rem';
  t.style.bottom='1rem';
  t.style.background='#212529';
  t.style.color='#fff';
  t.style.padding='.5rem .75rem';
  t.style.borderRadius='8px';
  t.style.zIndex='1055';
  document.body.appendChild(t);
  setTimeout(()=>t.remove(),1200);
}
</script>
